/*
    (c) 2002 - 2022 Juergen Nagel, Jan Hansen
    Northwest German Forest Research Station (https://www.nw-fva.de), 
    Grätzelstr. 2, 37079 Göttingen, Germany
    E-Mail: Jan.Hansen@nw-fva.de
 
    This file is part of the TreeGrOSS libraray.

    TreeGrOSS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    TreeGrOSS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with TreeGrOSS. If not, see http://www.gnu.org/licenses/.
*/

package treegross.base;

import java.lang.reflect.InvocationTargetException;
import java.util.logging.Level;
import java.util.logging.Logger;
import treegross.random.RandomNumber;

/**
 * TreeGrOSS : class tree defines the trees for class stand
 */
public class Tree implements Cloneable {

    /**
     * threshold value of solid wood (above diam. 7cm)
     */
    public final static double SW_THRESHOLD = 7.0;

    public long universalID = 0;
    /**
     * species Code according to Lower Saxony
     */
    public int code;
    /**
     * tree number
     */
    public String no;
    /**
     * age
     */
    public int age;
    /**
     * if living -1, else the year when died or taken out
     */
    public int out;
    /**
     * 0=standing, 1= fallen, 2=thinned, 3=harvested
     */
    public int outtype;
    /**
     * diameter, height, volume
     */
    public double d, h, v;
    /**
     * crown base and crown width
     */
    public double cb, cw;
    /**
     * relative tree position coordinates , z=vertical height of ground
     */
    public double x, y, z;
    /**
     * crop tree
     */
    public boolean crop;
    /**
     * temporary crop tree
     */
    public boolean tempcrop;
    /**
     * habitat tree
     */
    public boolean habitat;
    /**
     * competition index c66 c66xy c66c and c66cxy
     */
    public double c66, c66c, c66xy, c66cxy;
    /**
     * weight factor for concentric sample plots
     */
    public double fac;
    /**
     * site index
     */
    public double si;
    /**
     * reference to species
     */
    public Species sp;
    /**
     * reference to stand
     */
    public Stand st;
    /**
     * tree layer 1=upper, 2=middel, 3=lower
     */
    public int layer;
    /**
     * last diameter amd height increment
     */
    public double bhdinc, hinc;
    /**
     * year, tree grew into stand
     */
    public int year;
    /**
     * 0= unknown, 1=planting, 2=natural regeneration
     */
    public int origin;
    /**
     * remarks
     */
    public String remarks;
    public double hMeasuredValue;
    /**
     * Erlös pro Baum in Euro: getProceeds (get from
     * auxilliaries.SingletreeAsset
     */
    public double proceeds;
    /**
     * erntekostenfreier Erlös pro Baum in Euro: getPwoh (get from
     * auxilliaries.TreeValudation
     */
    public double pwohc;
    /**
     * Stammholzanteil in EFm: getSharelog (get from auxilliaries.TreeValudation
     */
    public double sharelog;
    /**
     * Kosten pro Baum in Euro: getCosts (get from auxilliaries.TreeValudation
     */
    public double costs;
    /**
     * X-Holzanteil: getShareXtimber (get from auxilliaries.TreeValudation
     */
    public double shareXtimber;
    /**
     * Totholz Zersetzungsgrad
     */
    public int zGrad = 0;
    /**
     * Volume of deadwood
     */
    public double volumeDeadwood = 0.0;
    /**
     * Volume of deadwood nature conservation
     */
    public double volumeDeadwoodConservation = 0.0;
    /**
     * Volume of harvested
     */
    public double volumeHarvested = 0.0;
    /**
     * Degree of Decay
     */
    public double degreeOfDecay = 0;
    /**
     * maximum number of neighbor trees
     */
    public int maxNeighbor = 15;
    /**
     * neighbor tree indices in radius of 2*crowthwidth
     */
    public int[] neighbor = new int[maxNeighbor];
    /**
     * number of neighbor trees
     */
    public int nNeighbor;
    /**
     * group
     */
    public int group;
    public int ihpot;
// für experimetal version
    /**
     * width and height of light crown
     */
    public double cbLightCrown = -9;
    /**
     * width and height of light crown
     */
    public double cwLightCrown = -9;
    /**
     * year of removal in reality
     */
    int yearOfRemovalinReality = 0;
// for Viswin 
    /* Ober- u. Unterstand */ public int ou = 0;
    /**
     * mortality reason 1= thinned or harvested, 2 = dry and standing, 3 = wind
     * throw, 4= other
     */
    public int mortalityReason = 0;
    //for ried
    /**
     * index of vitality
     */
    public double vitality = 1;
    /**
     * Measure 1=best to 0=worst of quality, used in Silviculture
     */
    public double quality = 1.0;
    /**
     * RandomSlope: Tree individual offset to diameter growth
     */
    public double randomSlope = 0.0;
    /**
     * RandomIntercept: Tree individual offset to diameter growth
     */
    public double randomIntercept = 0.0;

    public boolean outBySkidtrail = false;
    public int bioMass = 0; // kg   

    private final static Logger LOGGER = Logger.getLogger(Tree.class.getName());

    /**
     * empty tree contructor
     */
    public Tree() {
    }

    /**
     * Fill tree with data, fille missing information with negative value
     *
     * @param codex
     * @param nox
     * @param cbx
     * @param outx
     * @param outtypex
     * @param dx
     * @param agex
     * @param cwx
     * @param hx
     * @param six
     * @param facx
     * @param xx
     * @param tempCropTreex
     * @param zx
     * @param cropTreex
     * @param yx
     * @param habitatTreex
     * @param treeLayerx
     * @param volumeDeadwoodx
     * @param remarksx
     */
    public Tree(int codex, String nox, int agex, int outx, int outtypex, double dx, double hx, double cbx, double cwx,
            double six, double facx, double xx, double yx, double zx, boolean cropTreex, boolean tempCropTreex,
            boolean habitatTreex, int treeLayerx, double volumeDeadwoodx, String remarksx) {
        code = codex;
        no = nox;
        out = outx;
        outtype = outtypex;
        d = dx;
        h = hx;
        age = agex;
        cb = cbx;
        cw = cwx;
        si = six;
        fac = facx;
        x = xx;
        y = yx;
        z = zx;
        habitat = habitatTreex;
        crop = cropTreex;
        tempcrop = tempCropTreex;
        layer = treeLayerx;
        volumeDeadwood = volumeDeadwoodx;
        remarks = remarksx;
        quality = 0.5;
        randomSlope = 0.0;
        randomIntercept = 0.0;
    }

    /**
     * This function clones a Tree. The fields Stand and SpeciesDef will not be
     * cloned!!! Stand and SpeciesDef must be added after cloning a Tree. If
     * Tree.clone() is used in Stand.clone() the new cloned Stand must be added
     * to the new cloned tree by writing adiditonal code!
     *
     * @return a clone of this <code>Tree</code>
     */
    @Override
    public Tree clone() {
        Tree clone = new Tree();
        clone.universalID = this.universalID;
        clone.age = this.age;
        clone.bhdinc = this.bhdinc;
        clone.c66 = this.c66;
        clone.c66c = this.c66c;
        clone.c66cxy = this.c66cxy;
        clone.c66xy = this.c66xy;
        clone.cb = this.cb;
        clone.cbLightCrown = this.cbLightCrown;
        //clone.ci= new Double(this.ci);
        clone.code = this.code;
        clone.costs = this.costs;
        clone.crop = this.crop;
        clone.cw = this.cw;
        clone.cwLightCrown = this.cwLightCrown;
        clone.d = this.d;
        clone.fac = this.fac;
        clone.h = this.h;
        clone.hMeasuredValue = this.hMeasuredValue;
        clone.habitat = this.habitat;
        clone.hinc = this.hinc;
        clone.layer = this.layer;
        clone.no = this.no;
        clone.origin = this.origin;
        clone.ou = this.ou;
        clone.out = this.out;
        clone.outtype = this.outtype;
        clone.proceeds = this.proceeds;
        clone.pwohc = this.pwohc;
        if (remarks != null) {
            clone.remarks = this.remarks;
        } else {
            clone.remarks = "";
        }
        clone.shareXtimber = this.shareXtimber;
        clone.sharelog = this.sharelog;
        clone.si = this.si;
        clone.sp = sp.clone();
        //clone.st has to be added in the super clone call
        clone.v = this.v;
        clone.x = this.x;
        clone.y = this.y;
        clone.year = this.year;
        clone.yearOfRemovalinReality = this.yearOfRemovalinReality;
        clone.z = this.z;
        clone.zGrad = this.zGrad;
        clone.volumeDeadwood = this.volumeDeadwood;
        clone.degreeOfDecay = this.degreeOfDecay;
        clone.quality = this.quality;
        clone.randomSlope = this.randomSlope;
        clone.randomIntercept = this.randomIntercept;
        clone.vitality = vitality;
        clone.outBySkidtrail = this.outBySkidtrail;
        clone.bioMass = this.bioMass;
        clone.group = this.group;
        return clone;
    }

    /**
     *
     * @return
     */
    public double calculateCw() {
        double cwerg;
        if (sp.spDef.crownwidthXML.getFunctionText().length() > 2 && d >= 7.0) {
            FunctionInterpreter fi = new FunctionInterpreter();
            cwerg = fi.getValueForTree(this, sp.spDef.crownwidthXML);
        } else {
            cwerg = 0.01;
        }
        //regeneration placeholder 
        if (d < SW_THRESHOLD) {
            double dmerk = this.d;
            double hmerk = this.h;
            double cbmerk = this.cb;
            this.d = SW_THRESHOLD;
            this.h = 8.0;
            this.cb = 2.0;
            FunctionInterpreter fi = new FunctionInterpreter();
            cwerg = fi.getValueForTree(this, sp.spDef.crownwidthXML);
            this.d = dmerk;
            this.h = hmerk;
            this.cb = cbmerk;

            if (no == null) {
                no = "";
            }
        }
        if (cwerg > 50) {
            if (st != null && st.debug) {
                LOGGER.log(Level.SEVERE,
                        "Calculate crown width ({0}/{1}/{2}): cw unrealistic {3} using old cw: {4}",
                        new Object[]{st.standname, no, d, cwerg, cw});
            }
            cwerg = cw;
        }
        return cwerg;
    }

    /**
     *
     * @param hx
     * @return
     */
    public double calculateCwAtHeight(double hx) {
        double erg;
        if (d >= SW_THRESHOLD) {
            double cwAtHx = 0.0;
            double h66 = h - 2.0 * (h - cb) / 3.0;
            if (hx < h && hx > h66) {
                //cwAtHx = Math.sqrt(1.5 * Math.pow((cw / 2.0), 2.0) * (h - hx) / (h - cb));
                cwAtHx = Math.sqrt(1.5 * ((cw * 0.5) * (cw * 0.5)) * (h - hx) / (h - cb));
            }
            if (hx <= h66 && hx > cb) {
                cwAtHx = cw * 0.5;
            }
            erg = 2.0 * cwAtHx;
        } // else we are looking at the small trees representing a layer so they get crown width
        else {
            if (hx <= h) {
                //by jhansen: potential division by zero ???!!!
                double dCb = h - cb;
                if (dCb != 0) {
                    erg = (h - hx) * cw / (h - cb);
                } else {
                    //ToDo: check this
                    erg = (h - hx) * cw;
                }
            } else {
                erg = 0.0;
            }
            // erg = cw;
        }
        return erg;
    }

    /**
     *
     * @return
     */
    public double calculateCb() {
        double cberg = 0.0;
        if (sp.spDef.crownbaseXML.getFunctionText().length() > 4 && d >= SW_THRESHOLD) {
            FunctionInterpreter fi = new FunctionInterpreter();
            if (sp.h100 < 1.3 || Double.isNaN(sp.h100)) {
                sp.h100 = h;
            }
            cberg = fi.getValueForTree(this, sp.spDef.crownbaseXML);
        }
        if (d < SW_THRESHOLD) {
            cberg = h / 2.0;
        }
        if (cberg < 0.01) {
            cberg = 0.01;
        }
        if (Double.isNaN(cberg)) {
            boolean errout = false;
            if (st != null) {
                if (st.debug) {
                    errout = true;
                }
            }
            if (errout) {
                LOGGER.log(Level.WARNING, "Calculated crown base is NaN for: {0}/{1}\n\t{2} dbh: {3} height: {4} sp.h100: {5}.",
                        new Object[]{st.standname, no, sp.spDef.crownbaseXML, d, h, sp.h100});
            }
            cberg = h / 2.0;
        }
        return cberg;
    }

    /**
     *
     * @return
     */
    public double calculateVolume() {
        double erg = 0.0;
        if (d > 3.0 && h > 3.0) {
            FunctionInterpreter fi = new FunctionInterpreter();
            erg = fi.getValueForTree(this, sp.spDef.volumeFunctionXML);
            if (erg < 0.0) {
                erg = 0.0;
            }
            if (Double.isNaN(erg)) {
                erg = 0.0;
            }
        }
        return erg;
    }

    /**
     *
     * @return
     */
    public double calculateDecay() {
        double erg;
        FunctionInterpreter fi = new FunctionInterpreter();
        erg = fi.getValueForTree(this, sp.spDef.decayXML);
        if (erg < 0.0) {
            erg = 0.0;
        }
        if (erg < 1.0) {
            volumeDeadwood = v * erg;
        }
        return erg;
    }

    /**
     *
     * @return
     */
    public double calculateSiteIndex() {
        double erg;
        FunctionInterpreter fi = new FunctionInterpreter();
        erg = fi.getValueForTree(this, sp.spDef.siteindexXML);
        return erg;
    }

    /**
     *
     * @return
     */
    public double calculateMaxBasalArea() {
        double erg;
        FunctionInterpreter fi = new FunctionInterpreter();
        erg = fi.getValueForTree(this, sp.spDef.maximumDensityXML);
        return erg;
    }

    /**
     *
     */
    public void ageBasedMortality() {
        double ageindex = (1.0 * age / (1.0 * sp.spDef.maximumAge)) - 1.0;
        if (ageindex > st.random.nextUniform()) {
            out = st.year;
            outtype = 1;
        }
    }

    /**
     * get moderate thinning factor for maximum basal area from setting
     *
     * @return the moderate thinning factor
     */
    public double getModerateThinningFactor() {
        double tfac = 1.0;
        if (sp.spDef.moderateThinning.length() > 4) {
            String zeile = sp.spDef.moderateThinning;
            if (zeile.contains(";")) {
                // read in factors for height intervalls directly
                String[] tokens;
                tokens = zeile.split(";");
                // added by jhansen
                int end = tokens.length / 3;

                for (int i = 0; i < end; i++) {
                    double hu = Double.parseDouble(tokens[i * 3]);
                    double f = Double.parseDouble(tokens[i * 3 + 1]);
                    double ho = Double.parseDouble(tokens[i * 3 + 2]);
                    if (h >= hu && h < ho) {
                        tfac = f;
                        break;
                    }
                }
            } else {
                // read basal area function and calculate factor                
                FunctionInterpreter fi = new FunctionInterpreter();
                double basalarea = fi.getValueForTree(this, sp.spDef.moderateThinning);
                tfac = basalarea / this.calculateMaxBasalArea();
            }

        }
        return tfac;
    }

    public void setMissingData() {
        //if the first tree of a species is added there is no site index
        if (si <= -9.0 && sp.hbon <= 0.0) {
            if (sp.h100 <= 1.3 || Double.isNaN(sp.h100)) {
                sp.h100 = st.h100;
            }
            si = calculateSiteIndex();
        }
        if (si <= -9.0) {
            si = sp.hbon;
        }
        if (cb < 0.01) {
            cb = calculateCb();
        }
        if (cw < 0.01) {
            cw = calculateCw();
        }
        if (v <= 0.0) {
            v = calculateVolume();
        }
    }

    /**
     * grows a single tree for the length of up to 5 years, with and without
     * random effects. If the years is < 5 then the increment is set to
     * increment*years/5
     */
    void grow(int years, /*boolean randomEffects*/ RandomNumber rn) {
        //if(code== 999)
        //    return;     
        // trees with dbh >=150 cm do not grow anymore (WPKSKW)
        /*
         if (d > 150) {
         age += years;
         v = calculateVolume();
         return;
         }
         */
        FunctionInterpreter fi = new FunctionInterpreter();
        // Jugendwachstum, solange der BHD < 7.0 oder h < 1.3       
        if (d < SW_THRESHOLD) {
            double jh1;
            double jh2;
            double c66m = 2.0;
            Tree atree = new Tree();
            atree.sp = sp;
            atree.code = sp.code;
            atree.si = si;
            atree.age = 30;
            atree.no = no + " @ " + st.standname;
            jh1 = fi.getValueForTree(atree, sp.spDef.siteindexHeightXML);
            jh2 = jh1;
            jh1 = jh1 * ((Math.exp(age / 30.0) - 1.0) / (Math.exp(1.0) - 1.0));
            jh2 = jh2 * ((Math.exp((age + 5) / 30.0) - 1.0) / (Math.exp(1.0) - 1.0));
            hinc = jh2 - jh1;
            if (hinc < 0) {
                hinc = 0;
            }
            bhdinc = 0.0;
            double hd = 1.0;
            double dneu;
            if (sp.code < 200) {
                hd = 1.28;
                c66m = 1.2;
            } else if (sp.code >= 200 && sp.code < 300) {
                hd = 1.40;
                c66m = 2.2;
            } else if (sp.code >= 300 && sp.code < 400) {
                hd = 1.80;
                c66m = 1.5;
            } else if (sp.code >= 400 && sp.code < 500) {
                hd = 1.20;
                c66m = 1.2;
            } else if (sp.code >= 500 && sp.code < 600) {
                hd = 0.95;
                c66m = 1.8;
            } else if (sp.code >= 600 && sp.code < 700) {
                hd = 0.70;
                c66m = 1.9;
            } else if (sp.code >= 700 && sp.code < 800) {
                hd = 1.1;
                c66m = 1.2;
            } else if (sp.code >= 800 && sp.code < 900) {
                hd = 0.95;
                c66m = 1.0;
            }
            // The height growth is reduced for tree with high c66xy
            // 20% des Zuwachs durch Zufall
            double hmod = c66xy / c66m;
            if (hmod > 1.0) {
                hmod = 0.999;
            }
            if (hmod < 0.0) {
                hmod = 0.0;
            }
            hmod = 1.0 - (hmod * hmod);
            if (rn.randomOn) {
                //double eff = 0.2 * hinc;
                hinc = 0.9 * hinc + (0.2 * hinc) * rn.nextNormal(1);
            }
            dneu = (h + (years * hmod * (0.2 * hinc))) / hd;
            if (dneu > d) {
                d = dneu;
            }
        } else {
            /* Verwendung eines Rmodell
            if (sp.spDef.diameterIncrementXML.toString().contains("RFunction")){
                RConnection con = null;
                //RConnectionPool pool = null;
                try {
                    con = RConnectionPool.getInstance().borrow();                   
                    //System.out.println(con.isConnected());

                    bhdinc = getIg(con, this)/10000; 
                } catch (Exception e) {
                    e.printStackTrace();
                } finally {
                    if (con != null) {
                        RConnectionPool.getInstance().release(con);                       
                    }
                }
            } 
            else {
             */
            
            // basalarea increment in sqaure meters
            bhdinc = fi.getValueForTree(this, sp.spDef.diameterIncrementXML);
//            }
            boolean errout = st != null ? st.debug : false;
            if (Double.isInfinite(bhdinc) || Double.isNaN(bhdinc) || bhdinc < 0) {
                if (errout) {
                    LOGGER.log(Level.WARNING, "dbh increment is < 0, NaN or infinite for: {0}/{1}\n\tbhdinc: {2}\n\tc66cxy: {3} c66xy: {4}crown width {5}\n\tcrown base {6} species: {7} h100 for species: {8}", new Object[]{st.standname, no, bhdinc, c66cxy, c66xy, cw, cb, code, sp.h100});
                }
                bhdinc = 0;
            }

            //NormalDistributed ndis = new NormalDistributed();
            double effect;
            if (rn.randomOn) {
                if (bhdinc <= 0.0) {
                    bhdinc = 0.000001;
                }
                //effect = sp.spDef.diameterIncrementError*ndis.value(1.0);
                double e1 = 0.0;
                double e2 = 0.0;
                double e3 = 0.0;
                if (sp.spDef.diameterTreeErrorXML != null && ! !sp.spDef.diameterTreeErrorXML.isEmpty()) {
                    try {
                        String[] tec = sp.spDef.diameterTreeErrorXML.split(";");
                        if (tec.length >= 4) {
                            e1 = randomIntercept;
                            e2 = randomSlope * fi.getValueForTree(this, tec[2]);
                            e3 = rn.nextNormal() * fi.getValueForTree(this, tec[3]);
                        }
                    } catch (NumberFormatException e) {
                        LOGGER.log(Level.SEVERE, "Diameter Tree Error für Art {0} fehlerhaft !!!", code);
                    }
                }                
                if (e1 == 0.0 && e2 == 0.0 && e3 == 0.0) {
                    effect = sp.spDef.diameterIncrementError * rn.nextNormal(1);
                    bhdinc = Math.exp(Math.log(bhdinc) + effect);
                } else {
                    double bhdinc0 = bhdinc;
                    bhdinc = Math.exp(Math.log(bhdinc * 10000.0) + e1 + e2) / 10000.0;
                    bhdinc = (bhdinc * 10000.0 + e3) / 10000.0;   // dieser Effekt muss auf response-Ebene addiert werden!
                    if (st.debug) {
                        LOGGER.log(Level.INFO, "{0} ig {1} {2} {3} {4}  {5}", new Object[]{no, bhdinc0, e1, e2, e3, bhdinc});
                    }
                }
            }
            if (bhdinc < 0.0) {
                bhdinc = 0.0;
            }

            //bhdinc = 2.0 * Math.sqrt((Math.PI * Math.pow((d / 2.0), 2.0) + bhdinc * 10000.0) / Math.PI) - d;
            bhdinc = 2.0 * Math.sqrt((Math.PI * ((d * 0.5) * (d * 0.5)) + bhdinc * 10000.0) / Math.PI) - d;    
            //ToDo: check for negative increment
            if (bhdinc < 0.0) {
                bhdinc = 0.0;
            }

            // grow height (hinc????):
            double ihpot_l = fi.getValueForTree(this, sp.spDef.potentialHeightIncrementXML);
            
            // macht das hier Sinn???            
            /*if (h / sp.h100 >= 1.0) {
                hinc = 1.0;
            } else {
                hinc = sp.h100 / h;
            }*/            

            hinc = fi.getValueForTree(this, sp.spDef.heightIncrementXML);
            if (rn.randomOn) {
                effect = sp.spDef.heightIncrementError;
                hinc += effect * rn.nextNormal(1);
            }

            if (hinc > ihpot_l * 1.2) {
                hinc = ihpot_l * 1.2;   //ihpot*1.2 is max
            }
        }

        //no negative height growth allowed
        if (hinc < 0) {
            hinc = 0.0;
        }

        double ts = 0.2;
        if (st.timeStep > 0) {
            ts = 1.0 / st.timeStep;
        }

        h += years * hinc * ts;
        d += years * bhdinc * ts;
        age += years;
        v = calculateVolume();
    }
    
    void growBack(int years) {
        //System.out.println("grow back Tree");
        double ts = st.timeStep;
        if (out == st.year) {
            out = -1;
            outtype = 0;
        }
        RandomNumber rnOff = new RandomNumber(RandomNumber.OFF);
        if (out < 0) {
            Tree xtree;
            //explicitly set stand (not included in Tree.clone() -> Tree.grow(...) refers Stand.timeStep)
            xtree = this.clone();
            xtree.st = this.st;
            xtree.grow(years, rnOff);
            double dzvor = xtree.bhdinc;
            double hzvor = xtree.hinc;
            xtree = this.clone();
            xtree.st = this.st;
            xtree.h = xtree.h - years * hzvor / ts;
            xtree.d = xtree.d - years * dzvor / ts;
            xtree.age = xtree.age - years;
            FunctionInterpreter fi = new FunctionInterpreter();
            xtree.sp.h100 = xtree.sp.h100 - fi.getValueForTree(xtree, sp.spDef.potentialHeightIncrementXML);
            if (xtree.h < 1.3) {
                h = 4;
            }
            if (xtree.d < 1.0) {
                d = 1.0;
            }
            xtree.cw = xtree.calculateCw();
            xtree.cb = xtree.calculateCb();

            xtree.grow(years, rnOff);
            //System.out.println("grow back: d,h "+xtree.bhdinc+"  "+xtree.hinc);
            double newH = h - years * xtree.hinc / ts;
            if (newH < h) {
                h = newH;
            }
            double newD = d - years * xtree.bhdinc / ts;
            if (newD < d) {
                d = newD;
            }
            if (d < SW_THRESHOLD || h < 5.0) {
                d = 3.0;
                h = 4.0;
                cw = 2.0;
                cb = 2.0;
            }
            if (cb >= h) {
                cb = h - 0.2;
            }
            //System.out.println("grow back: cb,h "+cb+"  "+h);
            age -= years;
            v = calculateVolume();
        }
    }

    /**
     * update the crown base and width i.e. after growing of the tree
     */
    public void updateCrown() {
        if (d >= SW_THRESHOLD) {
            double cbnew;
            cbnew = calculateCb();
            if (cbnew > cb) {
                cb = cbnew;
            }
            // Crown base is not allowed to get a lower value
            cw = calculateCw();
        } //the little trees
        else {
            cb = h / 2.0;
            cw = calculateCw();
        }
    }

    /**
     * update the competition indices, should be called after growing, thinning,
     * and mortality
     */
    public void updateCompetition() {
        if (fac > 0.0 && out < 1) {
            //ToDo: check new approach with preloaded competition plugin class
            /*if (sp.spDef.competitionPlug != null) {
                //System.out.println("using plugin calss");
                //c66 = sp.spDef.competitionPlug.getc66(this);
                //c66c = sp.spDef.competitionPlug.getc66c(this);                
                sp.spDef.competitionPlug.replaceC66AndC66c(this);
                sp.spDef.competitionPlug.replaceC66xyAndC66cxy(this, cw);                
                if (lc66 != c66) {
                    System.out.println(lc66 + " <->" + c66);
                }
                if (lc66c != c66c) {
                    System.out.println(lc66c + " <->" + c66c);
                }
                
            } else*/ if (sp.spDef.competitionXML.length() > 1) {
                try {
                    String modelPlugIn = sp.spDef.competitionXML;
                    //PlugInCompetition comp = (PlugInCompetition) Class.forName(modelPlugIn).newInstance();
                    PlugInCompetition comp = (PlugInCompetition) Class.forName(modelPlugIn).getConstructor().newInstance();
                    //c66 = comp.getc66(this);
                    //c66c = comp.getc66c(this);
                    comp.replaceC66AndC66c(this);  //<- this does "getc66(this)" and "comp.getc66c(this)" in one step and reduces computation time
                    comp.replaceC66xyAndC66cxy(this, cw);
                } catch (ClassNotFoundException | IllegalAccessException | InstantiationException | NoSuchMethodException | SecurityException | InvocationTargetException e) {
                    LOGGER.log(Level.SEVERE, "ERROR in Class tree updateCompetition!", e);
                }
            } else {
                LOGGER.log(Level.SEVERE, "ERROR in Class tree updateCompetition! No COmpetition plugin specified for {0}", this.code);
            }
        } else {
            c66 = -99;
            c66c = -99.0;
            c66xy = -99;
            c66cxy = -99;
        }
    }

    /**
     * setGroup to assign the tree to an individual group
     *
     * @param group
     */
    public void setGroup(int group) {
        this.group = group;
    }
}
