/*
    (c) 2002 - 2022 Juergen Nagel, Jan Hansen
    Northwest German Forest Research Station (https://www.nw-fva.de), 
    Grätzelstr. 2, 37079 Göttingen, Germany
    E-Mail: Jan.Hansen@nw-fva.de
 
    This file is part of the TreeGrOSS libraray.

    TreeGrOSS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    TreeGrOSS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with TreeGrOSS. If not, see http://www.gnu.org/licenses/.
 */
package treegross.base;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URL;
import java.net.URLConnection;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;
import org.xml.sax.SAXException;
import treegross.tools.XmlTool;

/**
 * TreeGrOSS : TreegrossXML2.java version 7.5 18-Mar-2010 author	Juergen Nagel
 *
 * This is the 2nd format to define a forest stand by XML. The class can read
 * and write a treegross xml file
 *
 * http://www.nw-fva.de/~nagel/treegross/
 *
 */
public class TreegrossXML2 {

    private final static Logger LOGGER = Logger.getLogger(TreegrossXML2.class.getName());
    private boolean error;

    /**
     * Creates a new instance of TreegrossXML2
     */
    public TreegrossXML2() {
    }

    public boolean getError() {
        return error;
    }

    private Document makeXmlDocument(Stand st) {
        NumberFormat f;
        f = NumberFormat.getInstance(new Locale("en", "US"));
        f.setMaximumFractionDigits(2);
        f.setMinimumFractionDigits(2);
        DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder docBuilder;
        try {
            docBuilder = docFactory.newDocumentBuilder();
            Document doc = docBuilder.newDocument();
            ProcessingInstruction pi = doc.createProcessingInstruction("xml-stylesheet", "type=\"text/xsl\" href=\"treegross.xsl\"");
            doc.appendChild(pi);
            // add root element (stand)
            Element rootElt = doc.createElement("Bestand");
            doc.appendChild(rootElt);
            // add stand informations 
            if (st.nspecies > 0) {
                XmlTool.addChildString(rootElt, "Id", st.id);
                XmlTool.addChildString(rootElt, "Kennung", st.standname);
                XmlTool.addChildString(rootElt, "Allgemeines", " ");
                XmlTool.addChildString(rootElt, "Flaechengroesse_m2", Double.toString(st.size * 10000));
                XmlTool.addChildString(rootElt, "HauptbaumArtCodeStd", Integer.toString(st.sp[0].code));
                XmlTool.addChildString(rootElt, "HauptbaumArtCodeLokal", Integer.toString(st.sp[0].code));
                XmlTool.addChildString(rootElt, "AufnahmeJahr", Integer.toString(st.year));
                XmlTool.addChildString(rootElt, "AufnahmeMonat", Integer.toString(st.monat));
                XmlTool.addChildString(rootElt, "DatenHerkunft", st.datenHerkunft);
                XmlTool.addChildString(rootElt, "Standort", st.standort);
                XmlTool.addChildString(rootElt, "Hochwert_m", Double.toString(st.hochwert_m));
                XmlTool.addChildString(rootElt, "Rechtswert_m", Double.toString(st.rechtswert_m));
                XmlTool.addChildString(rootElt, "Hoehe_uNN_m", Double.toString(st.hoehe_uNN_m));
                XmlTool.addChildString(rootElt, "Exposition_Gon", Integer.toString(st.exposition_Gon));
                XmlTool.addChildString(rootElt, "Hangneigung_Prozent", Double.toString(st.hangneigungProzent));
                XmlTool.addChildString(rootElt, "Wuchsgebiet", st.wuchsgebiet);
                XmlTool.addChildString(rootElt, "Wuchsbezirk", st.wuchsbezirk);
                XmlTool.addChildString(rootElt, "Standortskennziffer", st.standortsKennziffer);
                XmlTool.addChildString(rootElt, "Status", Integer.toString(st.status));
                XmlTool.addChildString(rootElt, "Bemerkungen", st.bemerkungen);
            }
            // add all species
            for (int i = 0; i < st.nspecies; i++) {
                Element elt;
                elt = doc.createElement("Baumartencode");
                XmlTool.addChildString(elt, "Code", Integer.toString(st.sp[i].code));
                XmlTool.addChildString(elt, "deutscherName", st.sp[i].spDef.longName);
                XmlTool.addChildString(elt, "lateinischerName", st.sp[i].spDef.latinName);
                rootElt.appendChild(elt);
            }
            // add center point
            if (st.ncpnt > 0) {
                Element elt;
                elt = doc.createElement("Eckpunkt");
                XmlTool.addChildString(elt, "Nr", (st.center.no == null ? "n.d." : st.center.no));
                XmlTool.addChildString(elt, "RelativeXKoordinate_m", f.format(st.center.x));
                XmlTool.addChildString(elt, "RelativeYKoordinate_m", f.format(st.center.y));
                XmlTool.addChildString(elt, "RelativeBodenhoehe_m", f.format(st.center.z));
                rootElt.appendChild(elt);
            }
            // add corner points
            for (int i = 0; i < st.ncpnt; i++) {
                Element elt;
                elt = doc.createElement("Eckpunkt");
                XmlTool.addChildString(elt, "Nr", st.cpnt[i].no);
                XmlTool.addChildString(elt, "RelativeXKoordinate_m", f.format(st.cpnt[i].x));
                XmlTool.addChildString(elt, "RelativeYKoordinate_m", f.format(st.cpnt[i].y));
                XmlTool.addChildString(elt, "RelativeBodenhoehe_m", f.format(st.cpnt[i].z));
                rootElt.appendChild(elt);
            }
            // add trees:
            boolean lebend, entnommen;
            for (int i = 0; i < st.ntrees; i++) {
                Element elt;
                elt = doc.createElement("Baum");
                XmlTool.addChildString(elt, "Nr", Integer.toString(i + 1));
                XmlTool.addChildString(elt, "Kennung", st.tr[i].no);
                XmlTool.addChildString(elt, "BaumartcodeStd", "0");
                XmlTool.addChildString(elt, "BaumartcodeLokal", Integer.toString(st.tr[i].code));
                XmlTool.addChildString(elt, "Alter_Jahr", Integer.toString(st.tr[i].age));
                XmlTool.addChildString(elt, "BHD_mR_cm", f.format(st.tr[i].d));
                XmlTool.addChildString(elt, "Hoehe_m", f.format(st.tr[i].h));
                XmlTool.addChildString(elt, "Kronenansatz_m", f.format(st.tr[i].cb));
                XmlTool.addChildString(elt, "MittlererKronenDurchmesser_m", f.format(st.tr[i].cw));
                XmlTool.addChildString(elt, "SiteIndex_m", f.format(st.tr[i].si));
                XmlTool.addChildString(elt, "RelativeXKoordinate_m", f.format(st.tr[i].x));
                XmlTool.addChildString(elt, "RelativeYKoordinate_m", f.format(st.tr[i].y));
                XmlTool.addChildString(elt, "RelativeBodenhoehe_m", f.format(st.tr[i].z));
                lebend = (st.tr[i].out == -1);
                XmlTool.addChildString(elt, "Lebend", Boolean.toString(lebend));
                entnommen = (st.tr[i].outtype >= 2);
                XmlTool.addChildString(elt, "Entnommen", Boolean.toString(entnommen));
                XmlTool.addChildString(elt, "AusscheideMonat", "3");
                XmlTool.addChildString(elt, "AusscheideJahr", Integer.toString(st.tr[i].out));
                String grund;
                switch (st.tr[i].outtype) {
                    case 1:
                        grund = "Mortalität";
                        break;
                    case 2:
                        grund = "Durchforstung";
                        break;
                    case 3:
                        grund = "Ernte";
                        break;
                    default:
                        grund = "anderer";
                }
                XmlTool.addChildString(elt, "AusscheideGrund", grund);
                XmlTool.addChildString(elt, "ZBaum", Boolean.toString(st.tr[i].crop));
                XmlTool.addChildString(elt, "ZBaumtemporaer", Boolean.toString(st.tr[i].tempcrop));
                XmlTool.addChildString(elt, "HabitatBaum", Boolean.toString(st.tr[i].habitat));
                XmlTool.addChildString(elt, "KraftscheKlasse", "0");
                XmlTool.addChildString(elt, "Schicht", Integer.toString(st.tr[i].layer));
                f.setMaximumFractionDigits(4);
                f.setMinimumFractionDigits(4);
                XmlTool.addChildString(elt, "Flaechenfaktor", f.format(st.tr[i].fac));
                XmlTool.addChildString(elt, "Volumen_cbm", f.format(st.tr[i].v));
                XmlTool.addChildString(elt, "VolumenTotholz_cbm", f.format(st.tr[i].volumeDeadwood));
                XmlTool.addChildString(elt, "Bemerkung", st.tr[i].remarks);

                // die dynamischen Größen sind wichtig, um beim erneuten Laden des
                // XML Files konsistente Simulationsergebnisse zu erzielen (im Vergleich zu einer durchgehenden Simulation):
                //XmlTool.addChildString(elt, "C66", f.format(st.tr[i].fac)); // ??
                XmlTool.addChildString(elt, "C66c", f.format(st.tr[i].c66c));
                XmlTool.addChildString(elt, "C66cxy", f.format(st.tr[i].c66cxy));
                XmlTool.addChildString(elt, "hinc", f.format(st.tr[i].hinc));

                f.setMaximumFractionDigits(2);
                f.setMinimumFractionDigits(2);
                rootElt.appendChild(elt);
            }
            return doc;
        } catch (ParserConfigurationException ex) {
            LOGGER.log(Level.SEVERE, "treegross", ex);
        }
        return null;
    }

    public String getXmlAsString(Stand st) {
        Document doc = makeXmlDocument(st);
        if (doc != null) {
            try {
                DOMSource domSource = new DOMSource(doc);
                StringWriter writer = new StringWriter();
                StreamResult result = new StreamResult(writer);
                TransformerFactory tf = TransformerFactory.newInstance();
                Transformer transformer = tf.newTransformer();
                transformer.transform(domSource, result);
                return writer.toString();
            } catch (TransformerException ex) {
                LOGGER.log(Level.SEVERE, "treegross", ex);
            }
        }
        return null;
    }

    /**
     * save a treegross stand as xml file
     *
     * changes: jhansen, 02.12.2015: avoid usage of library jdom
     *
     *
     * @param st the Stand to save
     * @param fileName the full filename for xml file to write
     */
    public void saveAsXML(Stand st, String fileName) {
        Document doc = makeXmlDocument(st);
        if (doc != null) {
            try {
                // save to file
                TransformerFactory transformerFactory = TransformerFactory.newInstance();
                Transformer transformer = transformerFactory.newTransformer();
                DOMSource source = new DOMSource(doc);

                StreamResult result = new StreamResult(new File(fileName));
                // Output to console for testing
                //StreamResult result = new StreamResult(System.out);
                transformer.transform(source, result);
                //result.getOutputStream().close();

            } catch (TransformerException ex) {
                LOGGER.log(Level.SEVERE, "treegross", ex);
            }
        }
    }

    public Stand readTreegrossStand(Stand stl, URL url) {
        InputStream is = null;
        try {
            URLConnection urlcon = url.openConnection();
            is = urlcon.getInputStream();
            return readTreegrossStandFromStream(stl, is);
        } catch (IOException e) {
            LOGGER.log(Level.SEVERE, "treegross", e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException ex) {
                    LOGGER.log(Level.SEVERE, "treegross", ex);
                }
            }
        }
        return null;
    }

    public Stand readTreegrossStandFromStream(Stand stl, InputStream iss) {
        return readTreegrossStandFromStream(stl, iss, true);
    }

    /**
     * reads a treegross xml-File from InputStream
     *
     * changes
     * -------------------------------------------------------------------------
     * jhansen, 02.12.2015: avoid usage of library jdom
     * -------------------------------------------------------------------------
     *
     * @param stl TreeGrOSS Stand
     * @param iss
     * @param log
     * @return TreeGrOSS Stand
     */
    public Stand readTreegrossStandFromStream(Stand stl, InputStream iss, boolean log) {
        error = true;
        Stand st;
        if (stl != null) {
            st = stl;
        } else {
            st = new Stand();
        }
        try {
            Document d = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(iss);
            Element standElt = d.getDocumentElement();

            st.id = XmlTool.getChildText(standElt, "Id");
            st.addName(XmlTool.getChildText(standElt, "Kennung"));
            st.addsize(Double.parseDouble(XmlTool.getChildText(standElt, "Flaechengroesse_m2")) / 10000.0);
            st.monat = Integer.parseInt(XmlTool.getChildText(standElt, "AufnahmeMonat"));
            st.year = Integer.parseInt(XmlTool.getChildText(standElt, "AufnahmeJahr"));
            st.datenHerkunft = XmlTool.getChildText(standElt, "DatenHerkunft");
            st.standort = XmlTool.getChildText(standElt, "Standort");
            st.rechtswert_m = Double.parseDouble(XmlTool.getChildText(standElt, "Rechtswert_m", "-9"));
            st.hochwert_m = Double.parseDouble(XmlTool.getChildText(standElt, "Hochwert_m", "-9"));
            st.hoehe_uNN_m = Double.parseDouble(XmlTool.getChildText(standElt, "Hoehe_uNN_m", "-9"));
            st.exposition_Gon = Integer.parseInt(XmlTool.getChildText(standElt, "Exposition_Gon", "-9"));
            st.hangneigungProzent = Double.parseDouble(XmlTool.getChildText(standElt, "Hangneigung_Prozent", "0"));
            st.wuchsgebiet = XmlTool.getChildText(standElt, "Wuchsgebiet");
            st.wuchsbezirk = XmlTool.getChildText(standElt, "Wuchsbezirk");
            st.standortsKennziffer = XmlTool.getChildText(standElt, "Standortskennziffer");
            st.status = Integer.parseInt(XmlTool.getChildText(standElt, "Status", "0"));
            st.bemerkungen = XmlTool.getChildText(standElt, "Bemerkungen");

            st.ncpnt = 0;
            st.ntrees = 0;
            st.nspecies = 0;
            st.center.no = "undefined";

            NodeList cornerpoints = standElt.getElementsByTagName("Eckpunkt");
            Node n;
            if (cornerpoints != null && cornerpoints.getLength() > 0) {
                for (int i = 0; i < cornerpoints.getLength(); i++) {
                    n = cornerpoints.item(i);
                    String nrx = XmlTool.getChildText(n, "Nr");
                    if (nrx.contains("circle") || nrx.contains("polygon")) {
                        st.center.no = nrx;
                        st.center.x = Double.parseDouble(XmlTool.getChildText(n, "RelativeXKoordinate_m"));
                        st.center.y = Double.parseDouble(XmlTool.getChildText(n, "RelativeYKoordinate_m"));
                        st.center.z = Double.parseDouble(XmlTool.getChildText(n, "RelativeBodenhoehe_m"));
                    } else {
                        st.addcornerpoint(XmlTool.getChildText(n, "Nr"),
                                Double.parseDouble(XmlTool.getChildText(n, "RelativeXKoordinate_m")),
                                Double.parseDouble(XmlTool.getChildText(n, "RelativeYKoordinate_m")),
                                Double.parseDouble(XmlTool.getChildText(n, "RelativeBodenhoehe_m")));
                    }
                }
            } else {
                // if no corner point wa stored in xml create square with size equal to stand size
                double length = Math.sqrt(st.size * 10000);
                if (log) {
                    LOGGER.log(Level.WARNING, "No corner points stored in xml. Adding square coordinates (length={0})", length);
                }
                st.addcornerpoint("polygon_1", 0, 0, 0);
                st.addcornerpoint("polygon_2", 0, length, 0);
                st.addcornerpoint("polygon_3", length, length, 0);
                st.addcornerpoint("polygon_4", length, 0, 0);
            }

            NodeList trees = standElt.getElementsByTagName("Baum");
            for (int i = 0; i < trees.getLength(); i++) {
                n = trees.item(i);
                int out;// = -1 ;
                //if (Boolean.parseBoolean(baum.getChild("Entnommen").getText())==false) // wenn so dann muss hier flase nuss hier mit true abgeglichen werden
                out = Integer.parseInt(XmlTool.getChildText(n, "AusscheideJahr"));
                int outtype = 0;
                String ausGrund = XmlTool.getChildText(n, "AusscheideGrund");
                if (ausGrund.contains("Mort")) {
                    outtype = 1;
                } else if (ausGrund.contains("Durch")) {
                    outtype = 2;
                } else if (ausGrund.contains("Ernte")) {
                    outtype = 3;
                }
                try {
                    st.addXMLTree(Integer.parseInt(XmlTool.getChildText(n, "BaumartcodeLokal")),
                            XmlTool.getChildText(n, "Kennung"),
                            Integer.parseInt(XmlTool.getChildText(n, "Alter_Jahr")),
                            out, outtype,
                            Double.parseDouble(XmlTool.getChildText(n, "BHD_mR_cm")),
                            Double.parseDouble(XmlTool.getChildText(n, "Hoehe_m")),
                            Double.parseDouble(XmlTool.getChildText(n, "Kronenansatz_m")),
                            Double.parseDouble(XmlTool.getChildText(n, "MittlererKronenDurchmesser_m")),
                            Double.parseDouble(XmlTool.getChildText(n, "SiteIndex_m")),
                            Double.parseDouble(XmlTool.getChildText(n, "Flaechenfaktor")),
                            Double.parseDouble(XmlTool.getChildText(n, "RelativeXKoordinate_m")),
                            Double.parseDouble(XmlTool.getChildText(n, "RelativeYKoordinate_m")),
                            Double.parseDouble(XmlTool.getChildText(n, "RelativeBodenhoehe_m")),
                            Boolean.parseBoolean(XmlTool.getChildText(n, "ZBaum")),
                            Boolean.parseBoolean(XmlTool.getChildText(n, "ZBaumtemporaer")),
                            Boolean.parseBoolean(XmlTool.getChildText(n, "HabitatBaum")),
                            Integer.parseInt(XmlTool.getChildText(n, "Schicht")),
                            Double.parseDouble(XmlTool.getChildText(n, "VolumenTotholz_cbm")),
                            XmlTool.getChildText(n, "Bemerkung")
                    );
                    Tree addedTree = st.tr[st.ntrees - 1];
                    try {
                        Double c66c = Double.parseDouble(XmlTool.getChildText(n, "C66c", "0"));
                        addedTree.c66c = c66c;
                    } catch (NumberFormatException ne) {
                        LOGGER.log(Level.WARNING, "treegross", "ERROR while parseing numeric tree data for tree: " + XmlTool.getChildText(n, "Kennung") + " node: C66c");
                    }
                    try {
                        Double c66cxy = Double.parseDouble(XmlTool.getChildText(n, "C66cxy", "0"));
                        addedTree.c66cxy = c66cxy;
                    } catch (NumberFormatException ne) {
                        LOGGER.log(Level.WARNING, "treegross", "ERROR while parseing numeric tree data for tree: " + XmlTool.getChildText(n, "Kennung") + " node: C66cxy");
                    }
                    try {
                        Double hinc = Double.parseDouble(XmlTool.getChildText(n, "hinc", "0"));
                        addedTree.hinc = hinc;
                    } catch (NumberFormatException ne) {
                        LOGGER.log(Level.WARNING, "treegross", "ERROR while parseing numeric tree data for tree: " + XmlTool.getChildText(n, "Kennung") + " node: hinc");
                    }
                } catch (NumberFormatException ne) {
                    if (log) {
                        LOGGER.log(Level.SEVERE, "treegross", "ERROR while parseing numeric tree data for tree: " + XmlTool.getChildText(n, "Kennung"));
                        error = true;
                    }
                }
            }
            error = false;
        } catch (ParserConfigurationException | SAXException | IOException | SpeciesNotDefinedException ex) {
            if (log) {
                LOGGER.log(Level.SEVERE, "treegross", ex);
                error = true;
            }
        }
        return st;
    }

}
